package com.runner;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.xml.bind.DatatypeConverter;

import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPKeyConverter;

import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import util.NewPGPUtil;

public class BNPLSampleClient {

	public static void main(String[] args) {

	}

	public String encrypt(String base64EncodedPayloadString, String publicKeyToEncrypt)
			throws IOException, PGPException {

		PGPPublicKey pgpPublicKey = NewPGPUtil.readPublicKey(publicKeyToEncrypt);
		List<PGPPublicKey> pgpPublicKeys = new ArrayList<PGPPublicKey>();
		pgpPublicKeys.add(pgpPublicKey);

		// encrypt Payload - Use only HSBC Public Key
		String requestPayloadEncryptedBase64String = NewPGPUtil
				.encrpytRequestPayload(base64EncodedPayloadString, pgpPublicKeys, true, true, false);
		System.out.println("request payload encrypted Base64 String: " + requestPayloadEncryptedBase64String);
		return requestPayloadEncryptedBase64String;
	}
	
	public String createJwtToken(String clientOrProfileId, String base64EncodedEncryptPayloadString, String clientKeyId, String clientPrivateKey, String privateKeyPass) throws IOException, PGPException, NoSuchAlgorithmException {
		PGPPrivateKey privateKey = NewPGPUtil.readSecretKey(clientKeyId, clientPrivateKey, privateKeyPass);
		String token = createToken(clientOrProfileId, base64EncodedEncryptPayloadString, clientKeyId, new JcaPGPKeyConverter().getPrivateKey(privateKey));
		return token;
	}

	public String decrypt(String base64EncodedPayloadString, String privateKeyString, String privateKeyId, String privateKeyPass) throws PGPException, IOException {
		PGPPrivateKey pgpPrivateKey = NewPGPUtil.readSecretKey(privateKeyId, privateKeyString, privateKeyPass);
		String responseJsonString = NewPGPUtil.decryptResponseMessage( pgpPrivateKey, base64EncodedPayloadString);
		return responseJsonString;
	}

	
	/**
	 * createToken
	 *
	 * @param profileId                           profileId
	 * @param requestPayloadEncryptedBase64String requestPayloadEncryptedBase64String
	 * @param keyId                               keyId
	 * @param privateKey                          privateKey
	 * @return token
	 * @throws NoSuchAlgorithmException NoSuchAlgorithmException
	 */
	public static String createToken(String profileId, String requestPayloadEncryptedBase64String, String keyId,
			PrivateKey privateKey) throws NoSuchAlgorithmException {

		String hashingAlgorithm = "RSASHA256";
		// Generate unique ID for this API call
		String jwtId = UUID.randomUUID().toString();
		// Build JWT
		long nowSeconds = System.currentTimeMillis() / 1000;
		Map<String, Object> claims = new HashMap<String, Object>();
		claims.put("sub", profileId);
		claims.put("iat", nowSeconds);
		claims.put("aud", "HSBC.MKT");
		claims.put("jti", jwtId);
		String payloadHash = generateMessageDigest(requestPayloadEncryptedBase64String, "SHA-256");
		claims.put("payload_hash", payloadHash);
		System.out.println("payload_hash:" + payloadHash);
		claims.put("payload_hash_alg", hashingAlgorithm);

		Map<String, Object> header = new HashMap<String, Object>();
		header.put("typ", "JWT");
		header.put("kid", keyId);
		header.put("alg", SignatureAlgorithm.RS256.getValue());
		header.put("ver", "1.0");
		String token = Jwts.builder().setHeader(header).setClaims(claims).signWith(SignatureAlgorithm.RS256, privateKey)
				.compact();
		return token;
	}
	
	  // Helper method for getting message digest
	  private static String generateMessageDigest(String message, String hashingAlgorithm)
	      throws NoSuchAlgorithmException {
	    return DatatypeConverter
	        .printHexBinary(MessageDigest.getInstance(hashingAlgorithm).digest(message.getBytes()))
	        .toLowerCase();
	  }

}
